//-----------------------------------------------------------------------------
// Torque Game Engine
// 
// Copyright (c) 2001 GarageGames.Com
// Portions Copyright (c) 2001 by Sierra Online, Inc.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------

// Information extacted from the shape.
//
// Wheel Sequences
//    spring#        Wheel spring motion: time 0 = wheel fully extended,
//                   the hub must be displaced, but not directly animated
//                   as it will be rotated in code.
// Other Sequences
//    steering       Wheel steering: time 0 = full right, 0.5 = center
//    breakLight     Break light, time 0 = off, 1 = breaking
//
// Wheel Nodes
//    hub#           Wheel hub, the hub must be in it's upper position
//                   from which the springs are mounted.
//
// The steering and animation sequences are optional.
// The center of the shape acts as the center of mass for the car.

//-----------------------------------------------------------------------------

datablock ParticleData(TireParticle)
{
   textureName          = "~/data/shapes/car/dustParticle";
   dragCoefficient      = 2.0;
   gravityCoefficient   = -0.1;
   inheritedVelFactor   = 0.1;
   constantAcceleration = 0.0;
   lifetimeMS           = 1000;
   lifetimeVarianceMS   = 0;
   colors[0]     = "0.46 0.36 0.26 1.0";
   colors[1]     = "0.46 0.46 0.36 0.0";
   sizes[0]      = 0.50;
   sizes[1]      = 1.0;
};

datablock ParticleEmitterData(TireEmitter)
{
   ejectionPeriodMS = 10;
   periodVarianceMS = 0;
   ejectionVelocity = 1;
   velocityVariance = 1.0;
   ejectionOffset   = 0.0;
   thetaMin         = 5;
   thetaMax         = 20;
   phiReferenceVel  = 0;
   phiVariance      = 360;
   overrideAdvances = false;
   particles = "TireParticle";
};


//----------------------------------------------------------------------------

datablock WheeledVehicleTire(DefaultCarTire)
{
   // Tires act as springs and generate lateral and longitudinal
   // forces to move the vehicle. These distortion/spring forces
   // are what convert wheel angular velocity into forces that
   // act on the rigid body.
   shapeFile = "~/data/shapes/car/tire2.dts";
   friction = 1.5;

   // Spring that generates lateral tire forces
   lateralForce = 16000;
   lateralDamping = 1400;
   lateralRelaxation = 1;

   // Spring that generates longitudinal tire forces
   longitudinalForce = 16000;
   longitudinalDamping = 1400;
   longitudinalRelaxation = 1;
};

datablock WheeledVehicleSpring(DefaultCarSpring)
{
   // Wheel suspension properties
   length = 0.85;             // Suspension travel
   force = 2200;              // Spring force
   damping = 1600;            // Spring damping
   antiSwayForce = 3;         // Lateral anti-sway force
};

datablock WheeledVehicleData(DefaultCar : CarDamageProfile)
{
   category = "Vehicles";
   shapeFile = "~/data/shapes/car/car2.dts";

   maxDamage = 1.0;
   destroyedLevel = 1.0;
   
   minImpactSpeed = 29;
   speedDamageScale = 0.010;

   // Object Impact Damage (uses DamageType::Impact)
   collDamageThresholdVel = 23;
   collDamageMultiplier   = 0.030;

   cameraMaxDist = 5;
   cameraOffset = 1;

   maxSteeringAngle = 0.785;  // Maximum steering angle, should match animation
   integration = 4;           // Force integration time: TickSec/Rate
   tireEmitter = TireEmitter; // All the tires use the same dust emitter

   // Rigid Body
   mass = 200;
   bodyFriction = 0.6;
   bodyRestitution = 0.4;
   minImpactSpeed = 5;        // Impacts over this invoke the script callback
   softImpactSpeed = 5;       // Play SoftImpact Sound
   hardImpactSpeed = 15;      // Play HardImpact Sound

   // Engine
   engineTorque = 4000;       // Engine power
   engineBrake = 600;         // Braking when throttle is 0
   brakeTorque = 2000;        // When brakes are applied
   maxWheelSpeed = 30;        // Engine scale by current speed / max speed

   // Energy
   maxEnergy = 100;
   jetForce = 3000;
   minJetEnergy = 30;
   jetEnergyDrain = 2;

   // Sounds
//   jetSound = ScoutThrustSound;
//   engineSound = ScoutEngineSound;
//   squealSound = ScoutSquealSound;
//   softImpactSound = SoftImpactSound;
//   hardImpactSound = HardImpactSound;
//   wheelImpactSound = WheelImpactSound;

//   explosion = VehicleExplosion;
};

//-----------------------------------------------------------------------------

function WheeledVehicleData::create(%block)
{
   %obj = new WheeledVehicle() {
      dataBlock = %block;
   };
   return(%obj);
}

//-----------------------------------------------------------------------------

function WheeledVehicleData::onAdd(%this,%obj)
{
   // Setup the car with some defaults tires & springs
   for (%i = %obj.getWheelCount() - 1; %i >= 0; %i--) {
      %obj.setWheelTire(%i,DefaultCarTire);
      %obj.setWheelSpring(%i,DefaultCarSpring);
   }
}

function DefaultCar::onAdd(%this,%obj)
{
		parent::onAdd(%this,%obj);
	   %obj.mountImage(carbowimage,0);
      %obj.mountImage(carbow2image,1);
      %obj.setImageAmmo(0,1);
		%obj.setImageAmmo(1,1);
}

function DefaultCar::onDamage(%this, %obj)
{
   Parent::onDamage(%this, %obj);
}

datablock ShapeBaseImageData(CarbowImage)
{
   // Basic Item properties
   shapeFile = "~/data/shapes/car/carbow.dts";
   emap = true;

   // Specify mount point & offset for 3rd person, and eye offset
   // for first person rendering.
   mountPoint = 0;
   offset = "0 0 0";
   correctMuzzleVector = false;
   className = "WeaponImage";

   projectile = CrossbowProjectile;
   projectileType = Projectile;
	fireTimeout = 300;

   // Initial start up state
   stateName[0]                     = "Preactivate";
   stateTransitionOnLoaded[0]       = "Activate";
   stateTransitionOnNoAmmo[0]       = "Activate";

   // Activating the gun.  Called when the weapon is first
   // mounted and there is ammo.
   stateName[1]                     = "Activate";
   stateTransitionOnTimeout[1]      = "Ready";
   stateTimeoutValue[1]             = 0.6;
   stateSequence[1]                 = "Activate";

   // Ready to fire, just waiting for the trigger
   stateName[2]                     = "Ready";
   stateTransitionOnNoAmmo[2]       = "Activate";
   stateTransitionOnTriggerDown[2]  = "Fire";

   // Fire the weapon. Calls the fire script which does 
   // the actual work.
   stateName[3]                     = "Fire";
   stateTransitionOnTimeout[3]      = "Reload";
   stateTimeoutValue[3]             = 0.2;
   stateFire[3]                     = true;
   stateRecoil[3]                   = LightRecoil;
   stateAllowImageChange[3]         = false;
   stateSequence[3]                 = "Fire";
   stateScript[3]                   = "onFire";

   // Play the relead animation, and transition into
   stateName[4]                     = "Reload";
   stateTransitionOnNoAmmo[4]       = "Activate";
   stateTransitionOnTimeout[4]      = "Ready";
   stateTimeoutValue[4]             = 0.8;
   stateAllowImageChange[4]         = false;
   stateSequence[4]                 = "Reload";
   stateEjectShell[4]               = true;

   // No ammo in the weapon, just idle until something
   // shows up. Play the dry fire sound if the trigger is
   // pulled.
   stateName[5]                     = "Activate";
   stateTransitionOnAmmo[5]         = "Reload";
   stateSequence[5]                 = "Activate";
   stateTransitionOnTriggerDown[5]  = "DryFire";

   // No ammo dry fire
   stateName[6]                     = "DryFire";
   stateTimeoutValue[6]             = 1.0;
   stateTransitionOnTimeout[6]      = "Activate";
};

datablock ShapeBaseImageData(Carbow2Image : CarbowImage)
{
   mountPoint = 1;
};

function DefaultCar::onTrigger(%data, %obj, %trigger, %state)
{
   // data = datablock
   // obj = object number
   // trigger = 0 for "fire", 1 for "jump", 3 for "thrust"
   // state = 1 for firing, 0 for not firing
   if(%trigger == 0)
   {
      switch (%state) {
         case 0:
            %obj.fireWeapon = false;
            %obj.setImageTrigger(0, false);
            %obj.setImageTrigger(1, false);
         case 1:
            %obj.fireWeapon = true;
            if(%obj.nextWeaponFire == 0) {
               %obj.setImageTrigger(0, true);
               %obj.setImageTrigger(1, false);
            }
            else {
               %obj.setImageTrigger(0, false);
               %obj.setImageTrigger(1, true);
            }
      }
   }
}

function CarbowImage::onFire(%data,%obj,%slot)
{
   Parent::onFire(%data,%obj,%slot);
   %obj.nextWeaponFire = 1;
   schedule(%data.fireTimeout, 0, "fireNextGun", %obj);   
}

function Carbow2Image::onFire(%data,%obj,%slot)
{
   Parent::onFire(%data,%obj,%slot);
   %obj.nextWeaponFire = 0;
   schedule(%data.fireTimeout, 0, "fireNextGun", %obj);
}

function CarbowImage::onTriggerDown(%this, %obj, %slot)
{
}

function CarbowImage::onTriggerUp(%this, %obj, %slot)
{
}

function fireNextGun(%obj)
{
   if(%obj.fireWeapon)
   {
      if(%obj.nextWeaponFire == 0)
      {
         %obj.setImageTrigger(0, true);
         %obj.setImageTrigger(1, false);
      }
      else
      {
         %obj.setImageTrigger(0, false);
         %obj.setImageTrigger(1, true);
      }
   }
   else
   {
      %obj.setImageTrigger(0, false);
      %obj.setImageTrigger(1, false);
   }
}

function ShapeBaseImageData::onFire(%data, %obj, %slot)
{
	%projectile = %data.projectile;
	
	%muzzleVector = %obj.getMuzzleVector(%slot);
   %objectVelocity = %obj.getVelocity();
   
   %muzzleVelocity = VectorAdd(
      VectorScale(%muzzleVector, %projectile.muzzleVelocity),
      VectorScale(%objectVelocity, %projectile.velInheritFactor));
   
   %vehicle = 0;
   
       %p = new (%data.projectileType)() {
         dataBlock        = %data.projectile;
         initialVelocity  = %muzzleVelocity;
         initialDirection = %obj.getMuzzleVector(%slot);
         initialPosition  = %obj.getMuzzlePoint(%slot);
         sourceObject     = %obj;
         sourceSlot       = %slot;
         vehicleObject    = %vehicle;
      };
 
   if (isObject(%obj.lastProjectile) && %obj.deleteLastProjectile)
      %obj.lastProjectile.delete();

   %obj.lastProjectile = %p;
   %obj.deleteLastProjectile = %data.deleteLastProjectile;
   
   MissionCleanup.add(%p);
 
   return %p;
}